/*
 * Socket routines
 */

#include <stdio.h>
#ifndef _WIN32
#include <netdb.h>
#endif
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#ifndef _WIN32
#include <unistd.h>
#include <pthread.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#endif

#include "libfma.h"
#include "lf_internal.h"

/*
 * Mutex for gethostbyname
 */
static pthread_mutex_t _lf_connect_lock;

/*
 * Initialize pthread stuff
 */
void
lf_socket_init()
{
  pthread_mutex_init(&_lf_connect_lock, NULL);
}

/*
 * Finalize pthread stuff
 */
void
lf_socket_finalize()
{
  pthread_mutex_destroy(&_lf_connect_lock);
}

/*
 * Open an incoming socket and start listening on it
 */
int
lf_listen_on_socket(
  int port)
{
  int s;
  int rc;
  int one = 1;
  struct sockaddr_in my_addr;

  /* open the socket we will listen on */
  s = socket(AF_INET, SOCK_STREAM, 0);
  if (s == -1) LF_ERROR(("opening socket"));

  /* set up our address and port */
  memset(&my_addr, 0, sizeof(my_addr));
  my_addr.sin_family = AF_INET;
  my_addr.sin_addr.s_addr = (in_addr_t) htonl(INADDR_ANY);
  my_addr.sin_port = htons(port);

  /* set options for the socket */
  rc = setsockopt(s, SOL_SOCKET, SO_REUSEADDR, &one, sizeof(one));
  if (rc == -1) LF_ERROR(("setsockopt REUSEADDR"));
  rc = setsockopt(s, SOL_SOCKET, SO_KEEPALIVE, &one, sizeof(one));
  if (rc == -1) LF_ERROR(("setsockopt KEEPALIVE"));
  
  /* bind the socket to this address */
  rc = bind(s, (const struct sockaddr *)&my_addr, sizeof(my_addr));
  if (rc == -1) LF_ERROR(("bind socket to port %d", port));

  /* Initiate listening on the socket */
  rc = listen(s, 10);
  if (rc == -1) LF_ERROR(("listen"));

  return s;

 except:
  if (s != -1) close(s);
  return -1;
}

/*
 * Accept a new connection and set appropriate options
 */
int
lf_accept_connection(
  int s,
  void **who)
{
  int rc;
  int one = 1;
  struct sockaddr_in *addr;
  int ns;
  socklen_t len;

  addr = NULL;
  ns = -1;
  
  /* create a place to hold the address */
  len = sizeof(struct sockaddr_in);
  addr = (struct sockaddr_in *) calloc(1, len);
  if (addr == NULL) goto new_fail;

  /* create the new socket */
  ns = accept(s, (struct sockaddr *)addr, &len);
  if (ns == -1) goto new_fail;

  /* set its options */
  rc = setsockopt(ns, SOL_SOCKET, SO_KEEPALIVE, &one, sizeof(one));
  if (rc == -1) goto new_fail;

  /* return socket and address of connector */
  *who = (void *)addr;
  return ns;

 new_fail:
  if (ns != -1) close(ns);
  if (addr != NULL) free(addr);
  return -1;
}

/*
 * Open a socket connection to a remote listener
 */
int
lf_connect_to_host(
  char *hostname,
  int port)
{
  struct hostent *hp;
  struct sockaddr_in addr;
  int rc;
  int one = 1;
  int s;

  /* initialize for cleanup */
  pthread_mutex_lock(&_lf_connect_lock);
  s = -1;

  /* Translate hostname into an address */
  hp = gethostbyname(hostname);
  if (hp == NULL) {
    pthread_mutex_unlock(&_lf_connect_lock);
    LF_ERROR(("Error resolving hostname %s", hostname));
  }

  /* build full address */
  memcpy(&addr.sin_addr, hp->h_addr, hp->h_length);
  addr.sin_family = AF_INET;
  addr.sin_port = htons(port);

  /* done with the buffer */
  pthread_mutex_unlock(&_lf_connect_lock);

  /* open a socket for connecting */
  s = socket(AF_INET, SOCK_STREAM, 0);
  if (s == -1) LF_ERROR(("Error opening socket"));

  /* set its options */
  rc = setsockopt(s, SOL_SOCKET, SO_KEEPALIVE, &one, sizeof(one));
  if (rc == -1) LF_ERROR(("Error setting KEEPALIVE"));

  /* connect to the remote host */
  rc = connect(s, (struct sockaddr *)&addr, sizeof(struct sockaddr_in));
  if (rc == -1) LF_ERROR(("Error connecting to socket"));

  return s;

 except:
  if (s != -1) close(s);
  return -1;
}

/*
 * lf_read - a wrapper around reads since they can return short.
 */
int
lf_read(
  int d,
  void *buf,
  int len)
{
  int rc;
  int left;

  /*
   * Keep reading until we get everything the caller asked for, or
   * until 0 or error.  We error out on 0 becuase this is a blocking
   * read and should not return 0
   */
  left = len;
  while (left > 0) {
    rc = recv(d, (char*)buf + len - left, left, 0);
    if (rc == 0 || (rc == -1 && errno != EINTR
#if HAVE_DECL_ERESTART
		    && errno != ERESTART
#endif
		    && errno != EAGAIN)) {
      return -1;
    }
    left -= rc;
  }

  return len;
}

/*
 * lf_write - a wrapper around writes since they can return short.
 */
int
lf_write(
  int d,
  void *buf,
  int len)
{
  int rc;
  int left;

  /*
   * Keep writeing until we send everything the caller asked for, or
   * until 0 or error.  We error out on 0 becuase this is a blocking
   * write and should not return 0
   */
  left = len;
  while (left > 0) {
    rc = send(d, (char*)buf + len - left, left, 0);
    if (rc == 0 || (rc == -1 && errno != EINTR
#if HAVE_DECL_ERESTART
	  && errno != ERESTART
#endif
	  && errno != EAGAIN)) {
      return -1;
    }
    left -= rc;
  }

  return len;
}
